#!/usr/bin/env python
#
# Copyright Jon Nordby, 2010
#
# This library is free software; you can redistribute it and/or
# modify it under the terms of the GNU Library General Public
# License as published by the Free Software Foundation; either
# version 2 of the License, or (at your option) any later version.
#
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# Library General Public License for more details.
#
# You should have received a copy of the GNU Library General Public License
# along with this library; see the file COPYING.LIB.  If not, write to
# the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
# Boston, MA 02110-1301, USA.
#
# References:
# http://library.gnome.org/devel/integration-guide/stable/thumbnailer.html.en
# http://create.freedesktop.org/wiki/OpenRaster/File_Layout_Specification


import sys, os, zipfile, tempfile
from gtk import gdk

def get_pixbuf(png_data_str):
    """Return a gdk.Pixbuf with the contents of png_data_str"""
    # We create a temporary file in the filesystem because gdk.Pixbuf
    # does not let us get a pixbuf of the png file in an easy way without it
    temp_dir = tempfile.mkdtemp()
    temp_path = temp_dir + "thumb.png"

    temp_file = open(temp_path, "w")
    temp_file.write(png_data_str)
    temp_file.close()

    pixbuf = gdk.pixbuf_new_from_file(temp_path)
    os.remove(temp_path)
    os.rmdir(temp_dir)

    return pixbuf

def output_thumbnail(pixbuf, output_path, max_size):
    """Output a thumbnail of pixbuf to output_path.
    The thumbnail will be scaled so that neither dimensions exceeds max_size"""
    orig_w, orig_h = pixbuf.get_width(), pixbuf.get_height()

    if orig_w < max_size and orig_h < max_size:
        # No need for scaling
        pixbuf.save(outfile, 'png')
        return

    scale_factor = float(max_size)/max(orig_w, orig_h)
    new_w, new_h = int(orig_w*scale_factor), int(orig_h*scale_factor)
    scaled_pixbuf = pixbuf.scale_simple(new_w, new_h, gdk.INTERP_BILINEAR)
    scaled_pixbuf.save(outfile, 'png')

if __name__ == '__main__':
    try:
        progname, infile, outfile, size = sys.argv
    except ValueError:
        sys.exit('Usage: %s <Input> <Output> <Size>' % sys.argv[0])

    png_data = zipfile.ZipFile(infile).read('Thumbnails/thumbnail.png')
    pixbuf = get_pixbuf(png_data)
    output_thumbnail(pixbuf, outfile, int(size))

