/* $Id: thunar-file.h 17872 2005-09-27 22:14:26Z benny $ */
/*-
 * Copyright (c) 2005 Benedikt Meurer <benny@xfce.org>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef __THUNAR_FILE_H__
#define __THUNAR_FILE_H__

#include <thunar-vfs/thunar-vfs.h>

G_BEGIN_DECLS;

typedef struct _ThunarFileClass ThunarFileClass;
typedef struct _ThunarFile      ThunarFile;

#define THUNAR_TYPE_FILE            (thunar_file_get_type ())
#define THUNAR_FILE(obj)            (G_TYPE_CHECK_INSTANCE_CAST ((obj), THUNAR_TYPE_FILE, ThunarFile))
#define THUNAR_FILE_CLASS(klass)    (G_TYPE_CHECK_CLASS_CAST ((klass), THUNAR_TYPE_FILE, ThunarFileClass))
#define THUNAR_IS_FILE(obj)         (G_TYPE_CHECK_INSTANCE_TYPE ((obj), THUNAR_TYPE_FILE))
#define THUNAR_IS_FILE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), THUNAR_TYPE_FILE))
#define THUNAR_FILE_GET_CLASS(obj)  (G_TYPE_INSTANCE_GET_CLASS ((obj), THUNAR_TYPE_FILE, ThunarFileClass))

/**
 * ThunarFileDateType:
 * @THUNAR_FILE_DATE_ACCESSED : date of last access to the file.
 * @THUNAR_FILE_DATE_CHANGED  : date of last change to the file meta data or the content.
 * @THUNAR_FILE_DATE_MODIFIED : date of last modification of the file's content.
 *
 * The various dates that can be queried about a #ThunarFile. Note, that not all
 * #ThunarFile implementations support all types listed above. See the documentation
 * of the #thunar_file_get_date() method for details.
 **/
typedef enum
{
  THUNAR_FILE_DATE_ACCESSED,
  THUNAR_FILE_DATE_CHANGED,
  THUNAR_FILE_DATE_MODIFIED,
} ThunarFileDateType;

/**
 * ThunarFileIconState:
 * @THUNAR_FILE_ICON_STATE_DEFAULT : the default icon for the file.
 * @THUNAR_FILE_ICON_STATE_DROP    : the drop accept icon for the file.
 *
 * The various file icon states that are used within the file manager
 * views.
 **/
typedef enum
{
  THUNAR_FILE_ICON_STATE_DEFAULT,
  THUNAR_FILE_ICON_STATE_DROP,
} ThunarFileIconState;

#define THUNAR_FILE_EMBLEM_NAME_SYMBOLIC_LINK "emblem-symbolic-link"
#define THUNAR_FILE_EMBLEM_NAME_CANT_READ "emblem-noread"
#define THUNAR_FILE_EMBLEM_NAME_CANT_WRITE "emblem-nowrite"
#define THUNAR_FILE_EMBLEM_NAME_DESKTOP "emblem-desktop"

struct _ThunarFileClass
{
  GObjectClass __parent__;

  /* signals */
  void (*changed) (ThunarFile *file);
  void (*destroy) (ThunarFile *file);
};

struct _ThunarFile
{
  GObject __parent__;

  /*< private >*/
  ThunarVfsInfo *info;
  guint flags;
};

GType              thunar_file_get_type         (void) G_GNUC_CONST;

ThunarFile        *thunar_file_get_for_info     (ThunarVfsInfo          *info);
ThunarFile        *thunar_file_get_for_path     (ThunarVfsPath          *path,
                                                 GError                **error);

gboolean           thunar_file_has_parent       (const ThunarFile       *file);
ThunarFile        *thunar_file_get_parent       (const ThunarFile       *file,
                                                 GError                **error);

gboolean           thunar_file_execute          (ThunarFile             *file,
                                                 GdkScreen              *screen,
                                                 GList                  *path_list,
                                                 GError                **error);

gboolean           thunar_file_rename           (ThunarFile             *file,
                                                 const gchar            *name,
                                                 GError                **error);

GdkDragAction      thunar_file_accepts_drop     (ThunarFile             *file,
                                                 GList                  *path_list,
                                                 GdkDragAction           actions);

const gchar       *thunar_file_get_display_name (const ThunarFile       *file);
const gchar       *thunar_file_get_special_name (const ThunarFile       *file);

ThunarVfsPath     *thunar_file_get_path         (const ThunarFile       *file);
ThunarVfsMimeInfo *thunar_file_get_mime_info    (const ThunarFile       *file);
ThunarVfsFileTime  thunar_file_get_date         (const ThunarFile       *file,
                                                 ThunarFileDateType      date_type);
ThunarVfsFileType  thunar_file_get_kind         (const ThunarFile       *file);
ThunarVfsFileMode  thunar_file_get_mode         (const ThunarFile       *file);
ThunarVfsFileSize  thunar_file_get_size         (const ThunarFile       *file);

gchar             *thunar_file_get_date_string  (const ThunarFile       *file,
                                                 ThunarFileDateType      date_type);
gchar             *thunar_file_get_mode_string  (const ThunarFile       *file);
gchar             *thunar_file_get_size_string  (const ThunarFile       *file);

ThunarVfsVolume   *thunar_file_get_volume       (const ThunarFile       *file,
                                                 ThunarVfsVolumeManager *volume_manager);

ThunarVfsGroup    *thunar_file_get_group        (const ThunarFile       *file);
ThunarVfsUser     *thunar_file_get_user         (const ThunarFile       *file);

gboolean           thunar_file_is_executable    (const ThunarFile       *file);
gboolean           thunar_file_is_readable      (const ThunarFile       *file);
gboolean           thunar_file_is_renameable    (const ThunarFile       *file);
gboolean           thunar_file_is_writable      (const ThunarFile       *file);

GList             *thunar_file_get_actions      (ThunarFile             *file,
                                                 GtkWidget              *window);

GList             *thunar_file_get_emblem_names (const ThunarFile        *file);
const gchar       *thunar_file_get_icon_name    (const ThunarFile        *file,
                                                 ThunarFileIconState     icon_state,
                                                 GtkIconTheme           *icon_theme);

void               thunar_file_watch            (ThunarFile             *file);
void               thunar_file_unwatch          (ThunarFile             *file);

void               thunar_file_reload           (ThunarFile             *file);

void               thunar_file_changed          (ThunarFile             *file);
void               thunar_file_destroy          (ThunarFile             *file);

gboolean           thunar_file_is_hidden        (const ThunarFile       *file);

#define thunar_file_get_name(file)      (thunar_vfs_path_get_name (thunar_file_get_path ((file))))


/**
 * thunar_file_is_directory:
 * @file : a #ThunarFile.
 *
 * Checks whether @file refers to a directory.
 *
 * Return value: %TRUE if @file is a directory.
 **/
static inline gboolean
thunar_file_is_directory (const ThunarFile *file)
{
  g_return_val_if_fail (THUNAR_IS_FILE (file), FALSE);
  return (file->info->type == THUNAR_VFS_FILE_TYPE_DIRECTORY);
}

/**
 * thunar_file_is_home:
 * @file : a #ThunarFile.
 *
 * Checks whether @file refers to the users home directory.
 *
 * Return value: %TRUE if @file is the users home directory.
 **/
static inline gboolean
thunar_file_is_home (const ThunarFile *file)
{
  g_return_val_if_fail (THUNAR_IS_FILE (file), FALSE);
  return thunar_vfs_path_is_home (file->info->path);
}

/**
 * thunar_file_is_root:
 * @file : a #ThunarFile.
 *
 * Checks whether @file refers to the root directory.
 *
 * Return value: %TRUE if @file is the root directory.
 **/
static inline gboolean
thunar_file_is_root (const ThunarFile *file)
{
  g_return_val_if_fail (THUNAR_IS_FILE (file), FALSE);
  return thunar_vfs_path_is_root (file->info->path);
}


GList             *thunar_file_list_copy        (const GList            *file_list);
void               thunar_file_list_free        (GList                  *file_list);


G_END_DECLS;

#endif /* !__THUNAR_FILE_H__ */
