/* $Id: thunar-vfs.c 17769 2005-09-22 21:39:22Z benny $ */
/*-
 * Copyright (c) 2005 Benedikt Meurer <benny@xfce.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <thunar-vfs/thunar-vfs.h>
#include <thunar-vfs/thunar-vfs-listdir-job.h>
#include <thunar-vfs/thunar-vfs-transfer-job.h>
#include <thunar-vfs/thunar-vfs-unlink-job.h>
#include <thunar-vfs/thunar-vfs-alias.h>



static gint thunar_vfs_ref_count = 0;



/**
 * thunar_vfs_init:
 *
 * Initializes the ThunarVFS library.
 **/
void
thunar_vfs_init (void)
{
  if (g_atomic_int_exchange_and_add (&thunar_vfs_ref_count, 1) == 0)
    {
      /* initialize the path module */
      _thunar_vfs_path_init ();

      /* initialize the info module */
      _thunar_vfs_info_init ();

      /* initialize the jobs framework */
      _thunar_vfs_job_init ();
    }
}



/**
 * thunar_vfs_shutdown:
 *
 * Shuts down the ThunarVFS library.
 **/
void
thunar_vfs_shutdown (void)
{
  if (g_atomic_int_dec_and_test (&thunar_vfs_ref_count))
    {
      /* shutdown the jobs framework */
      _thunar_vfs_job_shutdown ();

      /* release the info module */
      _thunar_vfs_info_shutdown ();

      /* shutdown the path module */
      _thunar_vfs_path_shutdown ();
    }
}



/**
 * thunar_vfs_listdir:
 * @path  : the #ThunarVfsPath for the folder that should be listed.
 * @error : return location for errors or %NULL.
 *
 * Generates a #ThunarVfsListdirJob, which can be used to list the
 * contents of a directory (as specified by @path). If the creation
 * of the job failes for some reason, %NULL will be returned and
 * @error will be set to point to a #GError describing the cause.
 * Else the newly allocated #ThunarVfsListdirJob will be returned
 * and the caller is responsible to call g_object_unref().
 *
 * Note, that the returned job is launched right away, so you
 * don't need to call thunar_vfs_job_launch() on it.
 *
 * Return value: the newly allocated #ThunarVfsListdirJob or %NULL
 *               if an error occurs while creating the job.
 **/
ThunarVfsJob*
thunar_vfs_listdir (ThunarVfsPath *path,
                    GError       **error)
{
  g_return_val_if_fail (error == NULL || *error == NULL, NULL);

  /* allocate the job */
  return thunar_vfs_job_launch (thunar_vfs_listdir_job_new (path));
}



/**
 * thunar_vfs_copy:
 * @source_path_list : the list of #ThunarVfsPath<!---->s that should be copied.
 * @target_path      : the #ThunarVfsPath of the target directory.
 * @error            : return location for errors or %NULL.
 *
 * Generates a #ThunarVfsTransferJob, which can be used to copy
 * the files referenced by @source_path_list to the directory referred
 * to by @target_path. If the creation of the job failes for some reason,
 * %NULL will be returned and @error will be set to point to a #GError
 * describing the cause. Else the newly allocated #ThunarVfsTransferJob
 * will be returned and the caller is responsible to call
 * g_object_unref().
 *
 * Note, that the returned job is launched right away, so you don't
 * need to call thunar_vfs_job_launch() on it.
 *
 * Return value: the newly allocated #ThunarVfsTransferJob or %NULL
 *               if an error occurs while creating the job.
 **/
ThunarVfsJob*
thunar_vfs_copy (GList         *source_path_list,
                 ThunarVfsPath *target_path,
                 GError       **error)
{
  ThunarVfsJob *job;

  g_return_val_if_fail (error == NULL || *error == NULL, NULL);

  /* allocate/launch the job */
  job = thunar_vfs_transfer_job_new (source_path_list, target_path, FALSE, error);
  if (G_LIKELY (job != NULL))
    thunar_vfs_job_launch (job);

  return job;
}



/**
 * thunar_vfs_move:
 * @source_path_list : the list of #ThunarVfsPath<!---->s that should be moved.
 * @target_path      : the #ThunarVfsPath of the target directory.
 * @error            : return location for errors or %NULL.
 *
 * Generates a #ThunarVfsTransferJob, which can be used to move
 * the files referenced by @source_path_list to the directory referred
 * to by @target_path. If the creation of the job failes for some reason,
 * %NULL will be returned and @error will be set to point to a #GError
 * describing the cause. Else the newly allocated #ThunarVfsTransferJob
 * will be returned and the caller is responsible to call g_object_unref().
 *
 * Note, that the returned job is launched right away, so you don't
 * need to call thunar_vfs_job_launch() on it.
 *
 * Return value: the newly allocated #ThunarVfsTransferJob or %NULL
 *               if an error occurs while creating the job.
 **/
ThunarVfsJob*
thunar_vfs_move (GList         *source_path_list,
                 ThunarVfsPath *target_path,
                 GError       **error)
{
  ThunarVfsJob *job;

  g_return_val_if_fail (error == NULL || *error == NULL, NULL);

  /* allocate/launch the job */
  job = thunar_vfs_transfer_job_new (source_path_list, target_path, TRUE, error);
  if (G_LIKELY (job != NULL))
    thunar_vfs_job_launch (job);

  return job;
}



/**
 * thunar_vfs_unlink:
 * @path_list : a list of #ThunarVfsPath<!---->s, that should be unlinked.
 * @error     : return location for errors or %NULL.
 *
 * Generates a #ThunarVfsInteractiveJob, which can be used to unlink
 * all files referenced by the @path_list. If the creation of the job
 * failes for some reason, %NULL will be returned and @error will
 * be set to point to a #GError describing the cause. Else, the
 * newly allocated #ThunarVfsUnlinkJob will be returned, and the
 * caller is responsible to call g_object_unref().
 *
 * Note, that the returned job is launched right away, so you
 * don't need to call thunar_vfs_job_launch() on it.
 *
 * Return value: the newly allocated #ThunarVfsUnlinkJob or %NULL
 *               if an error occurs while creating the job.
 **/
ThunarVfsJob*
thunar_vfs_unlink (GList   *path_list,
                   GError **error)
{
  ThunarVfsJob *job;

  g_return_val_if_fail (error == NULL || *error == NULL, NULL);

  /* try to allocate the job */
  job = thunar_vfs_unlink_job_new (path_list, error);
  if (G_LIKELY (job != NULL))
    thunar_vfs_job_launch (job);

  return job;
}



#define __THUNAR_VFS_C__
#include <thunar-vfs/thunar-vfs-aliasdef.c>
