/* $Id: thunar-vfs-info.h 17594 2005-09-13 21:40:53Z benny $ */
/*-
 * Copyright (c) 2005 Benedikt Meurer <benny@xfce.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#if !defined (THUNAR_VFS_INSIDE_THUNAR_VFS_H) && !defined (THUNAR_VFS_COMPILATION)
#error "Only <thunar-vfs/thunar-vfs.h> can be included directly, this file may disappear or change contents."
#endif

#ifndef __THUNAR_VFS_INFO_H__
#define __THUNAR_VFS_INFO_H__

#include <thunar-vfs/thunar-vfs-mime-info.h>
#include <thunar-vfs/thunar-vfs-path.h>
#include <thunar-vfs/thunar-vfs-types.h>

G_BEGIN_DECLS;

#define THUNAR_VFS_TYPE_INFO (thunar_vfs_info_get_type ())

/**
 * ThunarVfsFileHint:
 * @THUNAR_VFS_FILE_HINT_ICON : The name of an icon or the path to
 *                              an image file, which should be used
 *                              to represent the file (e.g. the Icon
 *                              entry specified in a .desktop file).
 *
 * Additional hints that may be provided by the file system
 * backend.
 *
 * You can use thunar_vfs_info_get_hint() to query hints from a
 * #ThunarVfsInfo.
 **/
typedef enum {
  THUNAR_VFS_FILE_HINT_ICON,
  THUNAR_VFS_FILE_N_HINTS,
} ThunarVfsFileHint;

/**
 * ThunarVfsInfo:
 *
 * The #ThunarVfsInfo structure provides information about a file system
 * entity.
 **/
typedef struct
{
  /* File type */
  ThunarVfsFileType type;

  /* File permissions and special mode flags */
  ThunarVfsFileMode mode;

  /* File flags */
  ThunarVfsFileFlags flags;

  /* Owner's user id */
  ThunarVfsUserId uid;

  /* Owner's group id */
  ThunarVfsGroupId gid;

  /* Size in bytes */
  ThunarVfsFileSize size;

  /* time of last access */
  ThunarVfsFileTime atime;

  /* time of last modification */
  ThunarVfsFileTime mtime;

  /* time of last status change */
  ThunarVfsFileTime ctime;

  /* inode id */
  ThunarVfsFileInode inode;

  /* device id */
  ThunarVfsFileDevice device;

  /* file's mime type */
  ThunarVfsMimeInfo *mime_info;

  /* file's absolute path */
  ThunarVfsPath *path;

  /* file's display name (UTF-8) */
  gchar *display_name;

  /*< private >*/
  gchar **hints;
  gint    ref_count;
} ThunarVfsInfo;

GType                        thunar_vfs_info_get_type     (void) G_GNUC_CONST;

ThunarVfsInfo               *thunar_vfs_info_new_for_path (ThunarVfsPath       *path,
                                                           GError             **error) G_GNUC_MALLOC;

G_INLINE_FUNC ThunarVfsInfo *thunar_vfs_info_ref          (ThunarVfsInfo       *info);
void                         thunar_vfs_info_unref        (ThunarVfsInfo       *info);

gboolean                     thunar_vfs_info_execute      (const ThunarVfsInfo *info,
                                                           GdkScreen           *screen,
                                                           GList               *path_list,
                                                           GError             **error);

gboolean                     thunar_vfs_info_rename       (ThunarVfsInfo       *info,
                                                           const gchar         *name,
                                                           GError             **error);

const gchar                 *thunar_vfs_info_get_hint     (const ThunarVfsInfo *info,
                                                           ThunarVfsFileHint    hint);

gboolean                     thunar_vfs_info_matches      (const ThunarVfsInfo *a,
                                                           const ThunarVfsInfo *b);

G_INLINE_FUNC void           thunar_vfs_info_list_free    (GList               *info_list);


/* inline functions implementations */
#if defined(G_CAN_INLINE) || defined(__THUNAR_VFS_INFO_C__)
/**
 * thunar_vfs_info_ref:
 * @info : a #ThunarVfsInfo.
 *
 * Increments the reference count on @info by 1 and
 * returns a pointer to @info.
 *
 * Return value: a pointer to @info.
 **/
G_INLINE_FUNC ThunarVfsInfo*
thunar_vfs_info_ref (ThunarVfsInfo *info)
{
  exo_atomic_inc (&info->ref_count);
  return info;
}

/**
 * thunar_vfs_info_list_free:
 * @info_list : a list #ThunarVfsInfo<!---->s.
 *
 * Unrefs all #ThunarVfsInfo<!---->s in @info_list and
 * frees the list itself.
 *
 * This method always returns %NULL for the convenience of
 * being able to do:
 * <informalexample><programlisting>
 * info_list = thunar_vfs_info_list_free (info_list);
 * </programlisting></informalexample>
 *
 * Return value: the empty list (%NULL).
 **/
G_INLINE_FUNC void
thunar_vfs_info_list_free (GList *info_list)
{
  GList *lp;
  for (lp = info_list; lp != NULL; lp = lp->next)
    thunar_vfs_info_unref (lp->data);
  g_list_free (info_list);
}
#endif /* G_CAN_INLINE || __THUNAR_VFS_INFO_C__ */


#if defined(THUNAR_VFS_COMPILATION)
void           _thunar_vfs_info_init          (void) G_GNUC_INTERNAL;
void           _thunar_vfs_info_shutdown      (void) G_GNUC_INTERNAL;
ThunarVfsInfo *_thunar_vfs_info_new_internal  (ThunarVfsPath *path,
                                               const gchar   *absolute_path,
                                               GError       **error) G_GNUC_INTERNAL;
#endif

G_END_DECLS;

#endif /* !__THUNAR_VFS_INFO_H__ */
